/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2005 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__regcache.h"
#include "mx__fops.h"
#include "mx_util.h"

#if (MX_OS_LINUX && MX_RDMAWIN_CACHE) || MX_OS_UDRV
#include <malloc.h>
#include <syscall.h>
#include <limits.h>

void *mx___mremap(void *,size_t,size_t,int, void*) __asm__ ("__mremap");
void * mx___mremap (void *old_ptr, size_t old_size, size_t new_size, int maymove, void * extra)
{
  mx__regcache_clean(old_ptr,old_size);
  return (void*)syscall(__NR_mremap, old_ptr, old_size, new_size, maymove, extra);
}

void *mx_mremap(void *,size_t,size_t,int, void*) __asm__ ("mremap");
void * mx_mremap (void *old_ptr, size_t old_size, size_t new_size, int maymove, void *extra)
{
  return mx___mremap(old_ptr, old_size, new_size, maymove, extra);
}

int __munmap (void *old_ptr, size_t old_size)
{
  mx__regcache_clean(old_ptr,old_size);
  return syscall(__NR_munmap, old_ptr, old_size);
}

int munmap (void *old_ptr, size_t old_size)
{
  return __munmap(old_ptr, old_size);
}

extern void *__sbrk(intptr_t inc);

char *mx__init_brk;

void * sbrk (intptr_t inc)
{
#if !MX_OS_UDRV
  void *ptr;
  ptr =  __sbrk(inc);
  if (ptr && (intptr_t)ptr != -1 && inc) {
    mx__regcache_clean((inc >= 0 ? ptr : (char*)ptr - inc), inc);
    
  }
  return ptr;

#else
#ifndef PAGE_SIZE
#define PAGE_SIZE 4096
#endif
  static int fd = -1;
  static char *current_brk;
  static char *mapped_brk;
  int rc;
  char *new;
  if (fd < 0) {
    /* 
       we should not assume the libc is initialized
       moreover printf might call malloc 
       so have our own error handler
    */
    rc = mx__open(0,-1,&fd);
    if (rc) {
#define sbrk_error "MX sbrk replacement failed!!\n"
      write(2,sbrk_error, strlen(sbrk_error));
      abort();
    }
    current_brk = __sbrk(0);
    current_brk = (char*)MX_ROUND_UP((size_t)current_brk, PAGE_SIZE);
    mapped_brk = mx__init_brk = current_brk;
  }
  mx_always_assert(MX_ROUND_UP((size_t)__sbrk(0), PAGE_SIZE) == (size_t)mx__init_brk);
  new = (char*)MX_ROUND_UP((size_t)current_brk + inc, PAGE_SIZE);
  if (new > mapped_brk) {
    char * b;
    mx_always_assert(inc > 0);
    b = mx__mmap(mapped_brk, new - mapped_brk, fd, (mapped_brk - mx__init_brk) + 0x80000000UL, 0);
    if (b == MX_MAP_FAILED)
      return (void*)-1;
    mapped_brk = new;
    memset(current_brk,0,inc);
    current_brk = current_brk + inc;
    return current_brk - inc;
  } else {
    mx_always_assert(new >= mx__init_brk);
    if (inc > 0) {
      memset(current_brk, 0, inc);
    }
    current_brk += inc;
    return inc > 0 ? current_brk - inc : current_brk;
  }
#endif
}

static void *mx__morecore(intptr_t inc)
{
  void *res = sbrk(inc);
  return (intptr_t)res == -1 ? 0 : res;
}

extern int _DYNAMIC __attribute__((weak));

void dlfree(void *);
void *dlmalloc(size_t);
void *dlmemalign(size_t,size_t);
void *dlrealloc(void *,size_t);

static void free_hook(void *ptr, const void *caller)
{
  return dlfree(ptr);
}

static void *malloc_hook(size_t len, const void *caller)
{
  return dlmalloc(len);
}

static void *memalign_hook(size_t boundary,size_t size, const void * caller)
{
  return dlmemalign(boundary,size);
}

static void *realloc_hook(void *ptr,size_t len, const void * caller)
{
  return dlrealloc(ptr,len);
}

static void my_hook(void)
{
  char *s = getenv("MX_RCACHE");
  if ((s && s[0] == '1' && &_DYNAMIC != 0)
      || MX_OS_UDRV) {
    __free_hook = free_hook;
    __malloc_hook = malloc_hook;
    __memalign_hook = memalign_hook;
    __realloc_hook = realloc_hook;
  }
}

void (*__malloc_initialize_hook)() = my_hook;

#endif

int mx__regcache_works()
{
  static int works_ok = 0;
  static int already_called;
  if (already_called)
    return works_ok;
  already_called = 1;
#if MX_OS_LINUX && MX_RDMAWIN_CACHE
  free(malloc(1));
  /* catching address space changes works if
      - static linking (libc/malloc uses our __munmap/__morecore)
      - malloc != libc_malloc (detected by __malloc_hook != 0)
  */
  if (&_DYNAMIC == 0 || __malloc_hook != 0) {
    __morecore = mx__morecore;
    works_ok = 1;
    return 1;
  }
#elif MX_OS_WINNT && MX_RDMAWIN_CACHE
  works_ok = 1;
  return 1;
#endif
  works_ok = 0;
  return works_ok;
}

